// API 기본 URL 설정
const API_BASE_URL = "http://5.104.87.252:3000";

document.addEventListener('DOMContentLoaded', function() {
    // 인증 확인
    const token = localStorage.getItem('token');
    if (!token) {
        window.location.href = './login.html';
        return;
    }
    
    // UI 요소
    const userNameElement = document.getElementById('user-name');
    const userAvatarElement = document.getElementById('user-avatar');
    const logoutBtn = document.getElementById('logout-btn');
    const themeToggle = document.getElementById('theme-toggle');
    const sidebarToggle = document.getElementById('sidebar-toggle');
    const sidebar = document.getElementById('sidebar');
    
    const calendarTitle = document.getElementById('calendar-title');
    const calendarDays = document.getElementById('calendar-days');
    const prevMonthBtn = document.getElementById('prev-month');
    const nextMonthBtn = document.getElementById('next-month');
    const currentMonthBtn = document.getElementById('current-month');
    
    const upcomingAppointments = document.getElementById('upcoming-appointments');
    const selectedDateAppointments = document.getElementById('selected-date-appointments');
    const selectedDateTitle = document.getElementById('selected-date-title');
    const appointmentsForDate = document.getElementById('appointments-for-date');
    
    const showAddFormBtn = document.getElementById('show-add-form');
    const appointmentForm = document.getElementById('appointment-form');
    const addAppointmentForm = document.getElementById('add-appointment-form');
    const cancelAppointmentBtn = document.getElementById('cancel-appointment');
    const setReminderCheckbox = document.getElementById('set-reminder');
    const reminderOptions = document.getElementById('reminder-options');
    
    // 사용자 정보 로드
    let user = JSON.parse(localStorage.getItem('user')) || {};
    
    // 사용자 정보 표시
    userNameElement.textContent = user.firstName ? `${user.firstName} ${user.lastName || ''}` : user.username;
    if (user.firstName && user.lastName) {
        userAvatarElement.textContent = `${user.firstName.charAt(0)}${user.lastName.charAt(0)}`;
    } else if (user.username) {
        userAvatarElement.textContent = user.username.substring(0, 2).toUpperCase();
    }
    
    // 다크 모드 적용
    const currentTheme = localStorage.getItem('theme');
    if (currentTheme === 'dark') {
        document.body.setAttribute('data-theme', 'dark');
        themeToggle.innerHTML = '<i class="fas fa-sun"></i>';
    }
    
    // Sidebar toggle on mobile
    sidebarToggle.addEventListener('click', function() {
        sidebar.classList.toggle('active');
        document.body.classList.toggle('sidebar-open');
    });
    
    // Theme toggle
    themeToggle.addEventListener('click', function() {
        if (document.body.getAttribute('data-theme') === 'dark') {
            document.body.removeAttribute('data-theme');
            localStorage.setItem('theme', 'light');
            themeToggle.innerHTML = '<i class="fas fa-moon"></i>';
        } else {
            document.body.setAttribute('data-theme', 'dark');
            localStorage.setItem('theme', 'dark');
            themeToggle.innerHTML = '<i class="fas fa-sun"></i>';
        }
    });
    
    // Logout
    logoutBtn.addEventListener('click', function() {
        localStorage.removeItem('token');
        localStorage.removeItem('user');
        window.location.href = './login.html';
    });
    
    // 캘린더 관련 변수
    let currentDate = new Date();
    let currentMonth = currentDate.getMonth();
    let currentYear = currentDate.getFullYear();
    let appointments = [];
    
    // 달력 초기화
    renderCalendar(currentMonth, currentYear);
    
    // 다가오는 예약 로드
    loadUpcomingAppointments();
    
    // 월 변경 이벤트
    prevMonthBtn.addEventListener('click', function() {
        currentMonth--;
        if (currentMonth < 0) {
            currentMonth = 11;
            currentYear--;
        }
        renderCalendar(currentMonth, currentYear);
    });
    
    nextMonthBtn.addEventListener('click', function() {
        currentMonth++;
        if (currentMonth > 11) {
            currentMonth = 0;
            currentYear++;
        }
        renderCalendar(currentMonth, currentYear);
    });
    
    currentMonthBtn.addEventListener('click', function() {
        const today = new Date();
        currentMonth = today.getMonth();
        currentYear = today.getFullYear();
        renderCalendar(currentMonth, currentYear);
    });
    
    // 예약 폼 표시
    showAddFormBtn.addEventListener('click', function() {
        // 폼 초기화
        addAppointmentForm.reset();
        document.getElementById('appointment-id').value = '';
        document.getElementById('form-title').textContent = '새 예약 추가';
        
        // 현재 시간 + 1일을 기본값으로 설정
        const tomorrow = new Date();
        tomorrow.setDate(tomorrow.getDate() + 1);
        tomorrow.setHours(9, 0, 0, 0);
        
        // YYYY-MM-DDTHH:MM 형식으로 변환
        const formattedDate = tomorrow.toISOString().slice(0, 16);
        document.getElementById('appointment-date').value = formattedDate;
        
        // 폼 표시
        appointmentForm.style.display = 'block';
        
        // 페이지 상단으로 스크롤
        window.scrollTo({ top: 0, behavior: 'smooth' });
    });
    
    // 예약 추가 취소
    cancelAppointmentBtn.addEventListener('click', function() {
        appointmentForm.style.display = 'none';
    });
    
    // 알림 설정 표시/숨김
    setReminderCheckbox.addEventListener('change', function() {
        reminderOptions.style.display = this.checked ? 'block' : 'none';
    });
    
    // 예약 추가/수정 폼 제출
    addAppointmentForm.addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const appointmentId = document.getElementById('appointment-id').value;
        const doctorName = document.getElementById('doctor-name').value;
        const specialty = document.getElementById('specialty').value;
        const appointmentDate = document.getElementById('appointment-date').value;
        const location = document.getElementById('location').value;
        const notes = document.getElementById('appointment-notes').value || null;
        const setReminder = document.getElementById('set-reminder').checked;
        const reminderTime = setReminder ? document.getElementById('reminder-time').value : null;
        
        // 필수 필드 검증
        if (!doctorName || !specialty || !appointmentDate || !location) {
            alert('필수 필드를 모두 작성해 주세요');
            return;
        }
        
        // 예약 데이터 생성
        const appointmentData = {
            doctorName,
            specialty,
            appointmentDate,
            location,
            notes,
            setReminder: setReminder,
            reminderTime: parseInt(reminderTime)
        };
        
        try {
            let response;
            
            if (appointmentId) {
                // 예약 수정 (기본 URL 접두사 추가)
                response = await fetch(`${API_BASE_URL}/api/appointments/${appointmentId}`, {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${token}`
                    },
                    body: JSON.stringify(appointmentData)
                });
            } else {
                // 새 예약 추가 (기본 URL 접두사 추가)
                response = await fetch(`${API_BASE_URL}/api/appointments`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${token}`
                    },
                    body: JSON.stringify(appointmentData)
                });
            }
            
            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || '예약 저장 중 오류가 발생했습니다');
            }
            
            // 성공 메시지
            alert(appointmentId ? '예약이 성공적으로 업데이트되었습니다' : '예약이 성공적으로 추가되었습니다');
            
            // 폼 숨김
            appointmentForm.style.display = 'none';
            
            // 달력 및 예약 목록 새로고침
            renderCalendar(currentMonth, currentYear);
            loadUpcomingAppointments();
            
        } catch (error) {
            alert(error.message);
        }
    });
    
    // 달력 렌더링 함수
    async function renderCalendar(month, year) {
        // 달력 제목 업데이트
        calendarTitle.textContent = `${year}년 ${month + 1}월`;
        
        // 선택된 달의 첫날과 마지막날
        const firstDay = new Date(year, month, 1);
        const lastDay = new Date(year, month + 1, 0);
        
        // 첫날의 요일 (0: 일요일, 6: 토요일)
        const firstDayIndex = firstDay.getDay();
        
        // 이전 달의 마지막날
        const prevLastDay = new Date(year, month, 0);
        const prevLastDate = prevLastDay.getDate();
        
        // 달력에 표시할 전체 날짜 수 (이전 달, 현재 달, 다음 달)
        const daysInMonth = lastDay.getDate();
        const daysToShow = Math.ceil((firstDayIndex + daysInMonth) / 7) * 7;
        
        // 현재 날짜
        const today = new Date();
        
        // 이 달의 예약 로드
        await loadMonthAppointments(year, month);
        
        // HTML 생성
        let days = '';
        
        // 이전 달의 날짜
        for (let i = firstDayIndex - 1; i >= 0; i--) {
            const day = prevLastDate - i;
            const date = new Date(year, month - 1, day);
            const hasAppointments = checkAppointmentsForDate(date);
            
            days += `
                <div class="calendar-day other-month ${hasAppointments ? 'has-appointments' : ''}" data-date="${formatDateForAttribute(date)}">
                    <div class="day-number">${day}</div>
                    ${hasAppointments ? '<div class="appointment-indicator"></div>' : ''}
                </div>
            `;
        }
        
        // 현재 달의 날짜
        for (let i = 1; i <= daysInMonth; i++) {
            const date = new Date(year, month, i);
            const isToday = i === today.getDate() && month === today.getMonth() && year === today.getFullYear();
            const hasAppointments = checkAppointmentsForDate(date);
            
            days += `
                <div class="calendar-day ${isToday ? 'today' : ''} ${hasAppointments ? 'has-appointments' : ''}" data-date="${formatDateForAttribute(date)}">
                    <div class="day-number">${i}</div>
                    ${hasAppointments ? '<div class="appointment-indicator"></div>' : ''}
                </div>
            `;
        }
        
        // 다음 달의 날짜
        const remainingDays = daysToShow - (firstDayIndex + daysInMonth);
        for (let i = 1; i <= remainingDays; i++) {
            const date = new Date(year, month + 1, i);
            const hasAppointments = checkAppointmentsForDate(date);
            
            days += `
                <div class="calendar-day other-month ${hasAppointments ? 'has-appointments' : ''}" data-date="${formatDateForAttribute(date)}">
                    <div class="day-number">${i}</div>
                    ${hasAppointments ? '<div class="appointment-indicator"></div>' : ''}
                </div>
            `;
        }
        
        // 달력에 HTML 추가
        calendarDays.innerHTML = days;
        
        // 날짜 클릭 이벤트 리스너 추가
        document.querySelectorAll('.calendar-day').forEach(day => {
            day.addEventListener('click', function() {
                const dateAttr = this.getAttribute('data-date');
                if (dateAttr) {
                    showAppointmentsForDate(dateAttr);
                }
            });
        });
    }
    
    // 날짜별 예약 여부 확인
    function checkAppointmentsForDate(date) {
        const dateString = formatDateForChecking(date);
        return appointments.some(appointment => {
            const appointmentDate = new Date(appointment.appointmentDate);
            return formatDateForChecking(appointmentDate) === dateString;
        });
    }
    
    // 날짜별 예약 표시
    function showAppointmentsForDate(dateString) {
        const date = new Date(dateString);
        
        // 선택된 날짜의 예약 필터링
        const dateAppointments = appointments.filter(appointment => {
            const appointmentDate = new Date(appointment.appointmentDate);
            return formatDateForChecking(appointmentDate) === formatDateForChecking(date);
        });
        
        // 날짜 제목 설정
        selectedDateTitle.textContent = `${formatDate(date)}의 예약`;
        
        // 선택된 날짜의 예약이 있는 경우
        if (dateAppointments.length > 0) {
            let html = '';
            
            dateAppointments.forEach(appointment => {
                const appointmentTime = formatTime(new Date(appointment.appointmentDate));
                
                html += `
                    <div class="appointment-item">
                        <div class="appointment-header">
                            <h3 class="appointment-title">${appointment.doctorName}</h3>
                            <div class="appointment-actions">
                                <button class="edit-appointment" data-id="${appointment.id}" title="수정">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="delete-appointment" data-id="${appointment.id}" data-name="${appointment.doctorName}" title="삭제">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </div>
                        <div class="appointment-date">
                            <i class="fas fa-calendar-alt"></i> ${formatDate(new Date(appointment.appointmentDate))} ${appointmentTime}
                        </div>
                        <div class="appointment-details">
                            <div class="appointment-detail">
                                <i class="fas fa-stethoscope"></i> ${appointment.specialty}
                            </div>
                            <div class="appointment-detail">
                                <i class="fas fa-map-marker-alt"></i> ${appointment.location}
                            </div>
                        </div>
                        ${appointment.notes ? `
                            <div class="appointment-notes">
                                <i class="fas fa-sticky-note"></i> ${appointment.notes}
                            </div>
                        ` : ''}
                    </div>
                `;
            });
            
            appointmentsForDate.innerHTML = html;
            
            // 수정 버튼 이벤트 리스너
            document.querySelectorAll('.edit-appointment').forEach(button => {
                button.addEventListener('click', function() {
                    const id = this.getAttribute('data-id');
                    editAppointment(id);
                });
            });
            
            // 삭제 버튼 이벤트 리스너
            document.querySelectorAll('.delete-appointment').forEach(button => {
                button.addEventListener('click', function() {
                    const id = this.getAttribute('data-id');
                    const name = this.getAttribute('data-name');
                    deleteAppointment(id, name);
                });
            });
            
        } else {
            appointmentsForDate.innerHTML = '<p>이 날짜에 예약이 없습니다</p>';
        }
        
        // 선택된 날짜의 예약 섹션 표시
        selectedDateAppointments.style.display = 'block';
        
        // 선택된 날짜로 스크롤
        selectedDateAppointments.scrollIntoView({ behavior: 'smooth' });
    }
    
    // 월별 예약 로드
    async function loadMonthAppointments(year, month) {
        try {
            // 월의 첫날과 마지막날
            const firstDay = new Date(year, month, 1);
            const lastDay = new Date(year, month + 1, 0);
            
            // API URL 구성
            const url = `/api/appointments?from=${formatDateForAPI(firstDay)}&to=${formatDateForAPI(lastDay)}`;
            
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}${url}`, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('예약을 불러오는 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            appointments = data.data || [];
            console.log('월별 예약 데이터:', appointments);
            
        } catch (error) {
            console.error('월별 예약 로드 오류:', error);
            alert(error.message);
        }
    }
    
    // 다가오는 예약 로드
    async function loadUpcomingAppointments() {
        try {
            // 로딩 표시
            upcomingAppointments.innerHTML = `
                <div class="loading">
                    <div class="spinner"></div>
                </div>
            `;
            
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}/api/appointments?upcoming=true`, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('예약을 불러오는 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            const upcomingList = data.data || [];
            console.log('다가오는 예약 데이터:', upcomingList);
            
            // 다가오는 예약 렌더링
            if (upcomingList.length === 0) {
                upcomingAppointments.innerHTML = '<p>다가오는 예약이 없습니다</p>';
            } else {
                let html = '';
                
                upcomingList.forEach(appointment => {
                    const appointmentDate = new Date(appointment.appointmentDate);
                    const appointmentTime = formatTime(appointmentDate);
                    
                    html += `
                        <div class="appointment-item">
                            <div class="appointment-header">
                                <h3 class="appointment-title">${appointment.doctorName}</h3>
                                <div class="appointment-actions">
                                    <button class="edit-appointment" data-id="${appointment.id}" title="수정">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button class="delete-appointment" data-id="${appointment.id}" data-name="${appointment.doctorName}" title="삭제">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                            <div class="appointment-date">
                                <i class="fas fa-calendar-alt"></i> ${formatDate(appointmentDate)} ${appointmentTime}
                            </div>
                            <div class="appointment-details">
                                <div class="appointment-detail">
                                    <i class="fas fa-stethoscope"></i> ${appointment.specialty}
                                </div>
                                <div class="appointment-detail">
                                    <i class="fas fa-map-marker-alt"></i> ${appointment.location}
                                </div>
                            </div>
                            ${appointment.notes ? `
                                <div class="appointment-notes">
                                    <i class="fas fa-sticky-note"></i> ${appointment.notes}
                                </div>
                            ` : ''}
                        </div>
                    `;
                });
                
                upcomingAppointments.innerHTML = html;
                
                // 수정 버튼 이벤트 리스너
                document.querySelectorAll('.edit-appointment').forEach(button => {
                    button.addEventListener('click', function() {
                        const id = this.getAttribute('data-id');
                        editAppointment(id);
                    });
                });
                
                // 삭제 버튼 이벤트 리스너
                document.querySelectorAll('.delete-appointment').forEach(button => {
                    button.addEventListener('click', function() {
                        const id = this.getAttribute('data-id');
                        const name = this.getAttribute('data-name');
                        deleteAppointment(id, name);
                    });
                });
            }
            
        } catch (error) {
            upcomingAppointments.innerHTML = `<p>오류: ${error.message}</p>`;
        }
    }
    
    // 예약 수정 함수
    async function editAppointment(id) {
        try {
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}/api/appointments/${id}`, {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('예약 정보를 불러오는 중 오류가 발생했습니다');
            }
            
            const data = await response.json();
            const appointment = data.data;
            
            // 폼에 데이터 채우기
            document.getElementById('appointment-id').value = appointment.id;
            document.getElementById('doctor-name').value = appointment.doctorName;
            document.getElementById('specialty').value = appointment.specialty;
            
            // 날짜 및 시간 설정
            const dateTime = new Date(appointment.appointmentDate);
            // YYYY-MM-DDTHH:MM 형식으로 변환
            const formattedDateTime = dateTime.toISOString().slice(0, 16);
            document.getElementById('appointment-date').value = formattedDateTime;
            
            document.getElementById('location').value = appointment.location;
            document.getElementById('appointment-notes').value = appointment.notes || '';
            
            // 알림 설정
            const hasReminder = appointment.hasReminder || false;
            document.getElementById('set-reminder').checked = hasReminder;
            reminderOptions.style.display = hasReminder ? 'block' : 'none';
            
            if (hasReminder && appointment.reminderMinutes) {
                document.getElementById('reminder-time').value = appointment.reminderMinutes;
            }
            
            // 폼 제목 변경
            document.getElementById('form-title').textContent = '예약 편집';
            
            // 폼 표시
            appointmentForm.style.display = 'block';
            
            // 페이지 상단으로 스크롤
            window.scrollTo({ top: 0, behavior: 'smooth' });
            
        } catch (error) {
            alert(error.message);
        }
    }
    
    // 예약 삭제 함수
    async function deleteAppointment(id, name) {
        // 확인 대화상자
        if (!confirm(`${name} 의사와의 예약을 삭제하시겠습니까? 이 작업은 취소할 수 없습니다.`)) {
            return;
        }
        
        try {
            // 기본 URL 접두사 추가
            const response = await fetch(`${API_BASE_URL}/api/appointments/${id}`, {
                method: 'DELETE',
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });
            
            if (!response.ok) {
                throw new Error('예약을 삭제하는 중 오류가 발생했습니다');
            }
            
            // 성공 메시지
            alert('예약이 성공적으로 삭제되었습니다');
            
            // 달력 및 예약 목록 새로고침
            renderCalendar(currentMonth, currentYear);
            loadUpcomingAppointments();
            
            // 선택된 날짜의 예약이 표시되고 있다면 다시 로드
            if (selectedDateAppointments.style.display === 'block') {
                // 현재 선택된 날짜 가져오기
                const dateString = selectedDateTitle.textContent.replace('의 예약', '');
                const date = parseDate(dateString);
                if (date) {
                    showAppointmentsForDate(formatDateForAttribute(date));
                }
            }
            
        } catch (error) {
            alert(error.message);
        }
    }
    
    // 날짜 형식 함수
    function formatDate(date) {
        return date.toLocaleDateString('ko-KR', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            weekday: 'long'
        });
    }
    
    // 시간 형식 함수
    function formatTime(date) {
        return date.toLocaleTimeString('ko-KR', {
            hour: '2-digit',
            minute: '2-digit',
            hour12: true
        });
    }
    
    // 날짜를 속성용 문자열로 변환 (YYYY-MM-DD)
    function formatDateForAttribute(date) {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }
    
    // 날짜 비교용 문자열로 변환 (YYYY-MM-DD)
    function formatDateForChecking(date) {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }
    
    // API 요청용 날짜 형식 (YYYY-MM-DD)
    function formatDateForAPI(date) {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }
    
    // 문자열에서 날짜 파싱 (한국어 날짜 형식)
    function parseDate(dateString) {
        // '2025년 5월 19일 월요일' 형식에서 날짜 추출
        const match = dateString.match(/(\d{4})년\s+(\d{1,2})월\s+(\d{1,2})일/);
        if (match) {
            const year = parseInt(match[1]);
            const month = parseInt(match[2]) - 1; // 월은 0-11
            const day = parseInt(match[3]);
            return new Date(year, month, day);
        }
        return null;
    }
});