// API 기본 URL 설정
const API_BASE_URL = "http://5.104.87.252:3000";

// API 요청 함수들
const API = {
  // 기본 fetch 래퍼 함수
  async fetchWithAuth(url, options = {}) {
    const token = localStorage.getItem('token');
    
    // 기본 URL 접두사 추가
    const fullUrl = `${API_BASE_URL}${url}`;
    
    // 인증 토큰 추가
    const headers = {
      ...options.headers,
      'Authorization': token ? `Bearer ${token}` : ''
    };
    
    // 요청 전송
    const response = await fetch(fullUrl, {
      ...options,
      headers
    });
    
    return response;
  },
  
  // 로그인 API
  async login(credentials) {
    const response = await fetch(`${API_BASE_URL}/api/auth/login`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(credentials)
    });
    
    return response;
  },
  
  // 회원가입 API
  async register(userData) {
    const response = await fetch(`${API_BASE_URL}/api/auth/register`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(userData)
    });
    
    return response;
  },
  
  // 약품 관리 API
  async getMedications(queryParams = {}) {
    const queryString = Object.entries(queryParams)
      .filter(([_, value]) => value !== undefined && value !== null)
      .map(([key, value]) => `${key}=${encodeURIComponent(value)}`)
      .join('&');
    
    const url = `/api/medications${queryString ? `?${queryString}` : ''}`;
    return this.fetchWithAuth(url);
  },
  
  async getMedication(id) {
    return this.fetchWithAuth(`/api/medications/${id}`);
  },
  
  async createMedication(medicationData) {
    return this.fetchWithAuth('/api/medications', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(medicationData)
    });
  },
  
  async updateMedication(id, medicationData) {
    return this.fetchWithAuth(`/api/medications/${id}`, {
      method: 'PUT',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(medicationData)
    });
  },
  
  async deleteMedication(id) {
    return this.fetchWithAuth(`/api/medications/${id}`, {
      method: 'DELETE'
    });
  },
  
  // 약품 리마인더 API
  async getMedicationReminders(medicationId) {
    return this.fetchWithAuth(`/api/medications/${medicationId}/reminders`);
  },
  
  async createMedicationReminder(medicationId, reminderData) {
    return this.fetchWithAuth(`/api/medications/${medicationId}/reminders`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(reminderData)
    });
  },
  
  async updateReminder(reminderId, reminderData) {
    return this.fetchWithAuth(`/api/medications/reminders/${reminderId}`, {
      method: 'PUT',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(reminderData)
    });
  },
  
  async deleteReminder(reminderId) {
    return this.fetchWithAuth(`/api/medications/reminders/${reminderId}`, {
      method: 'DELETE'
    });
  },
  
  // 예약 관리 API
  async getAppointments(queryParams = {}) {
    const queryString = Object.entries(queryParams)
      .filter(([_, value]) => value !== undefined && value !== null)
      .map(([key, value]) => `${key}=${encodeURIComponent(value)}`)
      .join('&');
    
    const url = `/api/appointments${queryString ? `?${queryString}` : ''}`;
    return this.fetchWithAuth(url);
  },
  
  async getAppointment(id) {
    return this.fetchWithAuth(`/api/appointments/${id}`);
  },
  
  async createAppointment(appointmentData) {
    return this.fetchWithAuth('/api/appointments', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(appointmentData)
    });
  },
  
  async updateAppointment(id, appointmentData) {
    return this.fetchWithAuth(`/api/appointments/${id}`, {
      method: 'PUT',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(appointmentData)
    });
  },
  
  async deleteAppointment(id) {
    return this.fetchWithAuth(`/api/appointments/${id}`, {
      method: 'DELETE'
    });
  },
  
  // 알림 API
  async getReminders(queryParams = {}) {
    const queryString = Object.entries(queryParams)
      .filter(([_, value]) => value !== undefined && value !== null)
      .map(([key, value]) => `${key}=${encodeURIComponent(value)}`)
      .join('&');
    
    const url = `/api/reminders${queryString ? `?${queryString}` : ''}`;
    return this.fetchWithAuth(url);
  },
  
  // 건강 지표 API
  async getHealthMetrics(queryParams = {}) {
    const queryString = Object.entries(queryParams)
      .filter(([_, value]) => value !== undefined && value !== null)
      .map(([key, value]) => `${key}=${encodeURIComponent(value)}`)
      .join('&');
    
    const url = `/api/health-metrics${queryString ? `?${queryString}` : ''}`;
    return this.fetchWithAuth(url);
  },
  
  async getRecentHealthMetrics() {
    return this.fetchWithAuth('/api/health-metrics/recent');
  },
  
  // AI 음성 어시스턴트 API
  async aiQuery(queryData) {
    return this.fetchWithAuth('/api/ai-query', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(queryData)
    });
  }
};

// 외부에서 사용할 수 있도록 내보내기
export default API;